package tiger;

import java.awt.BorderLayout;
import java.awt.Toolkit;
import java.awt.event.*;
import java.io.*;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import javax.swing.*;

import routing.*;
import util.StandardFileFilter;

/**
 * Class for converting TIGER/Line Files into Network Files.
 * 
 * @version	2.00	18.08.2003	revised
 * @version	1.01	25.11.1999	adapted to the classes Nodes and Edges
 * @version	1.00	01.12.1998	first version
 * @author Thomas Brinkhoff
 */
public class TigerFileManager extends JFrame {

	/**
	 * Internal class for the event handling.
	 */
	class EventHandler implements ActionListener, ItemListener {
		// handling of action events
		public void actionPerformed (ActionEvent e) {
			statusLabel.setText("...");
			if (e.getSource() == exitMenuItem) 
				System.exit(0);
			else if (e.getSource() == readMenuItem) 
				readTigerFile();
			else if (e.getSource() == writeMenuItem) 
				writeNetworkFile();
			else if (e.getSource() == deleteMenuItem) {
				deleteNetwork();
				statusLabel.setText("Network is deleted.");
			}
			else if (e.getSource() == setTypeMenuItem) 
				setRecordType();
		};
		// handling of item events
		public void itemStateChanged (ItemEvent e) {
		};
	};

	/**
	 * The event handler.
	 */
	protected EventHandler eventHandler = new EventHandler();
	/**
	 * The content pane of the frame.
	 */
	protected JPanel frameContentPane = new JPanel();
	/**
	 * The read menu item.
	 */
	protected JMenuItem readMenuItem = new JMenuItem("Read Tiger File ...");
	/**
	 * The write menu item.
	 */
	protected JMenuItem writeMenuItem = new JMenuItem("Write Network Files");
	/**
	 * The delete menu item.
	 */
	protected JMenuItem deleteMenuItem = new JMenuItem("Delete Network");
	/**
	 * The exit menu item.
	 */
	protected JMenuItem exitMenuItem = new JMenuItem("Exit");
	/**
	 * The set record type menu item.
	 */
	protected JMenuItem setTypeMenuItem = new JMenuItem("Set Feature Class ...");
	/**
	 * Read file selector box.
	 */
	protected JFileChooser fileChooser;
	/**
	 * Status label.
	 */
	protected JLabel statusLabel = new JLabel("...");
	/**
	 * The standard path.
	 */
	protected static String filePath = "C:\\";
	/**
	 * The type of records extracted from the tiger file.
	 */
	protected String selectionType = "A";	// = roads
	/**
	 * The network.
	 */
	protected Hashtable nodes = new Hashtable();
	/**
	 * The base name of the current file.
	 */
	protected String baseName;
	/**
	 * Output stream for the edges.
	 */
	protected DataOutputStream edgeOut;
	/**
	 * Output stream for the nodes.
	 */
	protected DataOutputStream nodeOut;
	
	/**
	 * Thread loading a tiger file.
	 */
	class TigerLoader extends Thread {
		private String filename;	// name of the inputfile
		protected TigerLoader (String filename) {	// constructor
			this.filename = filename;
		}
		public void run() {
			try {
				baseName = filename.substring(0,filename.length()-4);
				// create input stream
				File inFile = new File(filename);
				long fileSize = inFile.length();
				InputStream in = new FileInputStream (inFile);
				if (filename.endsWith(".zip")) {
					ZipInputStream zis = new ZipInputStream(in);
					ZipEntry zEntry = zis.getNextEntry();
					fileSize = zEntry.getSize();
					in = zis;
				}
				LineNumberReader rin = new LineNumberReader (new InputStreamReader(in));
				int num = (int)(fileSize/ChainBasicData.getRecordSize());

				// Create output stream
				if (edgeOut == null)
					edgeOut =  new DataOutputStream (new BufferedOutputStream(new FileOutputStream (baseName+".edge")));
				if (nodeOut == null)
					nodeOut =  new DataOutputStream (new BufferedOutputStream(new FileOutputStream (baseName+".node")));
		
				// create network and read records
				ChainBasicData record = new ChainBasicData();
				int j = 0;
				while (true) {
					if (record.read(rin) == null)
						break;
					if (record.cfcc.startsWith(selectionType)) {
						// node 1
						int x = intoX(record.frLong);
						int y = intoY(record.frLat);
						long id = ((long)x)*Integer.MAX_VALUE+(long)y;
						Node frNode = new Node(id,x,y);
						Node hNode = (Node)nodes.get(frNode);
						if (hNode != null)
							frNode = hNode;
						else
							nodes.put(frNode,frNode);
						// node 2
						x = intoX(record.toLong);
						y = intoY(record.toLat);
						id = ((long)x)*Integer.MAX_VALUE+(long)y;
						Node toNode = new Node(id,x,y);
						hNode = (Node)nodes.get(toNode);
						if (hNode != null)
							toNode = hNode;
						else
							nodes.put(toNode,toNode);
						// edge
						Edge edge = new Edge(record.getID(),record.cfcc.charAt(1)-'1',frNode,toNode,null);
						edge.write (edgeOut);
					}

					j++;
					if (j % 1000 == 0)
						statusLabel.setText(j+" of "+num+" records have been read.");
				}
				in.close();
				statusLabel.setText("All records have been read.");
				writeMenuItem.setEnabled(true);
				deleteMenuItem.setEnabled(true);
			} catch (Exception ex) {
				ex.printStackTrace();
				JOptionPane.showMessageDialog (TigerFileManager.this, "File has not been read:\n"+ex, "OK", JOptionPane.ERROR_MESSAGE); 
			}
			readMenuItem.setEnabled(true);
		}
	}

	/**
	 * Thread writing the network files.
	 */
	class NetworkWriter extends Thread {
		protected NetworkWriter () {	// constructor
		}
		public void run() {
			try {
				// Store node file
				int i = 1;
				int num = nodes.size();;
				for (Enumeration e = nodes.elements(); e.hasMoreElements(); i++) {
					((Node)e.nextElement()).write (nodeOut);
					if (i % 1000 == 0)
						statusLabel.setText(i+" of "+num+" nodes have been stored.");
				}
				// delete network and close files
				deleteNetwork();
				statusLabel.setText("Network file are created.");
			} catch (Exception ex) {
				ex.printStackTrace();
				JOptionPane.showMessageDialog (TigerFileManager.this, "Files have not been written:\n"+ex, "OK", JOptionPane.ERROR_MESSAGE); 
			}
			readMenuItem.setEnabled(true);
		}
	}
	
/**
 * Constructor for MapConverter.
 * @throws HeadlessException
 */
public TigerFileManager() {
	super();
	setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);
	setSize(400, 300);
	setTitle("Tiger File Manager");

	// file selector box
	fileChooser = new JFileChooser(filePath);
	// define menu
	JMenuBar menuBar = new JMenuBar();
	setJMenuBar(menuBar);
	JMenu systemMenu = new JMenu("System");
	menuBar.add(systemMenu);
	readMenuItem.addActionListener(eventHandler);
	systemMenu.add(readMenuItem);
	writeMenuItem.addActionListener(eventHandler);
	systemMenu.add(writeMenuItem);
	writeMenuItem.setEnabled(false);
	systemMenu.add(new JSeparator());
	deleteMenuItem.addActionListener(eventHandler);
	systemMenu.add(deleteMenuItem);
	deleteMenuItem.setEnabled(false);
	systemMenu.add(new JSeparator());
	exitMenuItem.addActionListener(eventHandler);
	systemMenu.add(exitMenuItem);
	JMenu editMenu = new JMenu("Edit");
	menuBar.add(editMenu);
	setTypeMenuItem.addActionListener(eventHandler);
	editMenu.add(setTypeMenuItem);
	// define content pane
	frameContentPane.setLayout(new BorderLayout());
	frameContentPane.add(statusLabel,BorderLayout.SOUTH);
	setContentPane(frameContentPane);
}

/**
 * The main method: starts the application
 * @param  args  the argument array
 */
public static void main(String[] args) {
	try {
		/* Set native look and feel */
		UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
		/* Define path */
		if ((args != null) && (args.length > 0))
			filePath = args[0];
		/* Create the frame */
		TigerFileManager frame = new TigerFileManager();
		/* Calculate the screen size */
		java.awt.Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
		/* Center frame on the screen */
		java.awt.Dimension frameSize = frame.getSize();
		if (frameSize.height > screenSize.height)
			frameSize.height = screenSize.height;
		if (frameSize.width > screenSize.width)
			frameSize.width = screenSize.width;
		frame.setLocation((screenSize.width - frameSize.width) / 2, (screenSize.height - frameSize.height) / 2);
		// make visible
		frame.setVisible(true);
	} catch (Throwable exc) {
		System.err.println("TigerFileManager.main: "+exc);
	}
}

/**
 * Converts the longitude into a positive x-value.
 * @return  input value plus 180,000,000 (0 = 180 West)
 * @param  longitude  longitude in degree*1,000,000 
 */
private static int intoX (int longitude) {
	return longitude+180000000;
}

/**
 * Converts the latitude into a positive y-value.
 * @return  90,000,000 minus input value (0 = 90 North)
 * @param  latitude  latitude in degree*1,000,000 
 */
private static int intoY (int latitude) {
	return 90000000-latitude;
}

/**
 * Deletes network.
 */
public void deleteNetwork() {
	writeMenuItem.setEnabled(false);
	deleteMenuItem.setEnabled(false);
	nodes.clear();
	try {
		edgeOut.close();
		nodeOut.close();
	} catch (Exception ex) {
	}
	edgeOut = null;
	nodeOut = null;
}

/**
 * Reads the tiger file.
 */
public void readTigerFile() {
	// select input file
    fileChooser.setFileFilter(new StandardFileFilter(new String[]{"zip","rt1"},"tiger files"));
	int returnVal = fileChooser.showOpenDialog(this);
	if (returnVal != JFileChooser.APPROVE_OPTION)
		return;
	// determine tiger file name
	String selectedName = fileChooser.getSelectedFile().getPath();
	File f = new File(selectedName);
	if (!f.exists()) {
		JOptionPane.showMessageDialog (this, "File "+selectedName+" does not exist!", "OK", JOptionPane.ERROR_MESSAGE); 
		return;
	}
	// reads the file in a thread
	readMenuItem.setEnabled(false);
	writeMenuItem.setEnabled(false);
	deleteMenuItem.setEnabled(false);
	new TigerLoader(selectedName).start();
}

/**
 * Set record type.
 */
public void setRecordType() {
	String result = JOptionPane.showInputDialog("Define the feature class for selecting records:\n(A = roads, B = railways, H = hydrography\n A1 = primary highways and so on)", selectionType);
	if (result != null)
		selectionType = result;
}

/**
 * Writes the network files.
 */
public void writeNetworkFile() {
	readMenuItem.setEnabled(false);
	writeMenuItem.setEnabled(false);
	deleteMenuItem.setEnabled(false);
	new NetworkWriter().start();
}

}