package conversion;

import java.awt.BorderLayout;
import java.awt.Toolkit;
import java.awt.event.*;
import java.io.*;
import java.nio.channels.*;
import java.util.Enumeration;
import java.util.Hashtable;
import javax.swing.*;

import routing.*;
import util.StandardFileFilter;

import org.geotools.data.shapefile.dbf.*;
import org.geotools.data.shapefile.shp.*;
import com.vividsolutions.jts.geom.*;

/**
 * Class for converting Shapefiles into Network Files.
 * 
 * @version	2.00	28.10.2003	first version derived from the TigerFileManager
 * @author Thomas Brinkhoff
 */
public class ShapeNetworkFileManager extends JFrame {

	/**
	 * Internal class for the event handling.
	 */
	class EventHandler implements ActionListener, ItemListener {
		// handling of action events
		public void actionPerformed (ActionEvent e) {
			statusLabel.setText("...");
			if (e.getSource() == exitMenuItem) 
				System.exit(0);
			else if (e.getSource() == readMenuItem) 
				readShapeFile();
			else if (e.getSource() == writeMenuItem) 
				writeNetworkFile();
			else if (e.getSource() == deleteMenuItem) {
				deleteNetwork();
				statusLabel.setText("Network is deleted.");
			}
			else if (e.getSource() == setTypeMenuItem) 
				setResolution();
		};
		// handling of item events
		public void itemStateChanged (ItemEvent e) {
		};
	};

	/**
	 * The event handler.
	 */
	protected EventHandler eventHandler = new EventHandler();
	/**
	 * The content pane of the frame.
	 */
	protected JPanel frameContentPane = new JPanel();
	/**
	 * The read menu item.
	 */
	protected JMenuItem readMenuItem = new JMenuItem("Read Shapefile ...");
	/**
	 * The write menu item.
	 */
	protected JMenuItem writeMenuItem = new JMenuItem("Write Network Files");
	/**
	 * The delete menu item.
	 */
	protected JMenuItem deleteMenuItem = new JMenuItem("Delete Network");
	/**
	 * The exit menu item.
	 */
	protected JMenuItem exitMenuItem = new JMenuItem("Exit");
	/**
	 * The set record type menu item.
	 */
	protected JMenuItem setTypeMenuItem = new JMenuItem("Set Resolution of Network File ...");
	/**
	 * The store shape points menu item.
	 */
	protected JCheckBoxMenuItem storeShapePointsMenuItem = new JCheckBoxMenuItem("Store Shape Points");
	/**
	 * Read file selector box.
	 */
	protected JFileChooser fileChooser;
	/**
	 * Status label.
	 */
	protected JLabel statusLabel = new JLabel("...");
	/**
	 * The standard path.
	 */
	protected static String filePath = "C:\\";
	/**
	 * The network.
	 */
	protected Hashtable nodes = new Hashtable();
	/**
	 * The base name of the current file.
	 */
	protected String baseName;
	/**
	 * Output stream for the edges.
	 */
	protected DataOutputStream edgeOut;
	/**
	 * Output stream for the nodes.
	 */
	protected DataOutputStream nodeOut;
	/**
	 * The resolution of the network map
	 */
	protected int resolution = 30000;
	/** 
	 * Minimum x-coordinate of map.
	 */
	protected double minX;
	/** 
	 * Minimum y-coordinate of map.
	 */
	protected double maxY;
	/** 
	 * x-extend of map.
	 */
	private double dX;
	/** 
	 * y-extend map.
	 */
	private double dY;
	
	/**
	 * Thread loading a shape file.
	 */
	class ShapeLoader extends Thread {
		private String filename;	// name of the inputfile
		
		// constructor
		protected ShapeLoader (String filename) {
			this.filename = filename;
		}
		
		// stores a segment 
		protected void storeSegment (Point p1, Point p2, long edgeId, int edgeClass) {
			// node 1
			int x = intoX(p1.getX(),minX,dX);
			int y = intoY(p1.getY(),maxY,dY);
			long id = ((long)x)*Integer.MAX_VALUE+(long)y;
			Node frNode = new Node(id,x,y);
			Node hNode = (Node)nodes.get(frNode);
			if (hNode != null)
				frNode = hNode;
			else
				nodes.put(frNode,frNode);
			// node 2
			x = intoX(p2.getX(),minX,dX);
			y = intoY(p2.getY(),maxY,dY);
			id = ((long)x)*Integer.MAX_VALUE+(long)y;
			Node toNode = new Node(id,x,y);
			hNode = (Node)nodes.get(toNode);
			if (hNode != null)
				toNode = hNode;
			else
				nodes.put(toNode,toNode);
			// edge
			Edge edge = new Edge(edgeId,edgeClass,frNode,toNode,null);
			edge.write (edgeOut);
		}
		
		// select database column (-1: no selection)
		protected int selectDbColumn (DbaseFileReader dbr) {
			DbaseFileHeader dbh = dbr.getHeader();
			// determine name column
			int colNum = dbh.getNumFields();
			String[] colNames = new String[colNum];
			for (int i=0; i<colNum; i++)
				colNames[i] = dbh.getFieldName(i).toString();
			Object selectedValue = JOptionPane.showInputDialog(null, 
				"Choose class column", "Input", JOptionPane.INFORMATION_MESSAGE, null,
				colNames, colNames[0]);
			if (selectedValue == null)
				return -1;
			int selectedCol = 0;
			while ((selectedCol+1<colNum) && (selectedValue != colNames[selectedCol]))
				selectedCol++;
			return selectedCol;
		}
		
		// converts the database entry into edge class
		protected int toEdgeClass (Object dataEntry) {
			try {
				return Math.abs(Integer.parseInt(dataEntry.toString())) % 6;
			} catch (Exception ex) {
				return 3;
			}
		}
		
		// main method
		public void run() {
			try {
				baseName = filename.substring(0,filename.length()-4);
				// create input stream
				FileChannel in = new FileInputStream(filename).getChannel();
				ShapefileReader r = new ShapefileReader(in);
				ShapefileHeader header = r.getHeader();
				// test file type
				if (!header.getShapeType().isLineType()) {
					JOptionPane.showMessageDialog (ShapeNetworkFileManager.this, "File "+baseName+" does not contain lines!", "OK", JOptionPane.ERROR_MESSAGE); 
					r.close();
					readMenuItem.setEnabled(true);
					return;
				}
				// open database file
				FileChannel dbin = new FileInputStream(baseName+".dbf").getChannel();
				DbaseFileReader dbr = new DbaseFileReader(dbin);
				int classCol = selectDbColumn(dbr);
				// count number
				FileChannel iin = new FileInputStream(baseName+".shx").getChannel();
				IndexFile idxFile = new IndexFile(iin);
				int num = idxFile.getRecordCount();
				// set map extent
				if (dX == 0) {
					minX = header.minX();
					maxY = header.maxY();
					dX = header.maxX()-minX;
					dY = maxY-header.minY();
				}
				// create output stream
				if (edgeOut == null)
					edgeOut =  new DataOutputStream (new BufferedOutputStream(new FileOutputStream (baseName+".edge")));
				if (nodeOut == null)
					nodeOut =  new DataOutputStream (new BufferedOutputStream(new FileOutputStream (baseName+".node")));
		
				// create network and read records
				int j = 0;	// # records
				int k = 0;	// # segments
				while (r.hasNext()) {
					// process database entry
					Object[] data = dbr.readEntry();
					int edgeClass = classCol < 0 ? 3 : toEdgeClass(data[classCol]);
					Geometry shape = (Geometry) r.nextRecord().shape();
					// process lines
					if (shape instanceof MultiLineString) {
						MultiLineString lines = (MultiLineString) shape;
						for (int i=0; i<lines.getNumGeometries(); i++) {
							LineString line = (LineString)lines.getGeometryN(i);
							// case: line only as one segment
							if (! storeShapePointsMenuItem.getState()) {
								Point p1 = line.getStartPoint();
								Point p2 = line.getEndPoint();
								storeSegment(p1,p2,k,edgeClass);
								k++;
							}
							// case: all segments of line
							else {
								for (int pi=0; pi<line.getNumPoints()-1; pi++) {
									Point p1 = line.getPointN(pi);
									Point p2 = line.getPointN(pi+1);
									storeSegment(p1,p2,k,edgeClass);
									k++;
								}
							}
						}
					}
					else {
						System.err.println("Wrong class: "+shape.getClass().getName());
						break;
					}
					j++;
					if (j % 1000 == 0)
						statusLabel.setText(j+" of "+num+" records have been read.");
				}
				// close
				r.close();
				iin.close();
				
				statusLabel.setText("All records have been read.");
				writeMenuItem.setEnabled(true);
				deleteMenuItem.setEnabled(true);
			} catch (Exception ex) {
				ex.printStackTrace();
				JOptionPane.showMessageDialog (ShapeNetworkFileManager.this, "File has not been read:\n"+ex, "OK", JOptionPane.ERROR_MESSAGE); 
			}
			readMenuItem.setEnabled(true);
		}
	}

	/**
	 * Thread writing the network files.
	 */
	class NetworkWriter extends Thread {
		protected NetworkWriter () {	// constructor
		}
		public void run() {
			try {
				// Store node file
				int i = 1;
				int num = nodes.size();;
				for (Enumeration e = nodes.elements(); e.hasMoreElements(); i++) {
					((Node)e.nextElement()).write (nodeOut);
					if (i % 1000 == 0)
						statusLabel.setText(i+" of "+num+" nodes have been stored.");
				}
				// delete network and close files
				deleteNetwork();
				statusLabel.setText("Network file are created.");
			} catch (Exception ex) {
				ex.printStackTrace();
				JOptionPane.showMessageDialog (ShapeNetworkFileManager.this, "Files have not been written:\n"+ex, "OK", JOptionPane.ERROR_MESSAGE); 
			}
			readMenuItem.setEnabled(true);
		}
	}
	
/**
 * Constructor for MapConverter.
 * @throws HeadlessException
 */
public ShapeNetworkFileManager() {
	super();
	setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);
	setSize(400, 300);
	setTitle("Shapefile - Network File Manager");

	// file selector box
	fileChooser = new JFileChooser(filePath);
	// define menu
	JMenuBar menuBar = new JMenuBar();
	setJMenuBar(menuBar);
	JMenu systemMenu = new JMenu("System");
	menuBar.add(systemMenu);
	readMenuItem.addActionListener(eventHandler);
	systemMenu.add(readMenuItem);
	writeMenuItem.addActionListener(eventHandler);
	systemMenu.add(writeMenuItem);
	writeMenuItem.setEnabled(false);
	systemMenu.add(new JSeparator());
	deleteMenuItem.addActionListener(eventHandler);
	systemMenu.add(deleteMenuItem);
	deleteMenuItem.setEnabled(false);
	systemMenu.add(new JSeparator());
	exitMenuItem.addActionListener(eventHandler);
	systemMenu.add(exitMenuItem);
	JMenu editMenu = new JMenu("Edit");
	menuBar.add(editMenu);
	setTypeMenuItem.addActionListener(eventHandler);
	editMenu.add(setTypeMenuItem);
	editMenu.add(storeShapePointsMenuItem);
	// define content pane
	frameContentPane.setLayout(new BorderLayout());
	frameContentPane.add(statusLabel,BorderLayout.SOUTH);
	setContentPane(frameContentPane);
}

/**
 * The main method: starts the application
 * @param  args  the argument array
 */
public static void main(String[] args) {
	try {
		/* Set native look and feel */
		UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
		/* Define path */
		if ((args != null) && (args.length > 0))
			filePath = args[0];
		/* Create the frame */
		ShapeNetworkFileManager frame = new ShapeNetworkFileManager();
		/* Calculate the screen size */
		java.awt.Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
		/* Center frame on the screen */
		java.awt.Dimension frameSize = frame.getSize();
		if (frameSize.height > screenSize.height)
			frameSize.height = screenSize.height;
		if (frameSize.width > screenSize.width)
			frameSize.width = screenSize.width;
		frame.setLocation((screenSize.width - frameSize.width) / 2, (screenSize.height - frameSize.height) / 2);
		// make visible
		frame.setVisible(true);
	} catch (Throwable exc) {
		System.err.println("ShapeNetworkFileManager.main: "+exc);
	}
}

/**
 * Converts the longitude into a positive x-value.
 * @return  the x-value
 * @param  longitude  longitude in degree 
 * @param  min  the minimum value of data space
 * @param  diff  the extension of the data space
 */
private int intoX (double longitude, double min, double ext) {
	return (int)((longitude-min)*resolution/ext);
}

/**
 * Converts the latitude into a positive y-value.
 * @return  y-value
 * @param  latitude  latitude in degree 
 * @param  max  the maximum value of data space
 * @param  diff  the extension of the data space
 */
private int intoY (double latitude, double max, double ext) {
	return (int)((max-latitude)*resolution/ext);
}

/**
 * Deletes network.
 */
public void deleteNetwork() {
	writeMenuItem.setEnabled(false);
	deleteMenuItem.setEnabled(false);
	nodes.clear();
	try {
		edgeOut.close();
		nodeOut.close();
	} catch (Exception ex) {
	}
	edgeOut = null;
	nodeOut = null;
	dX = 0;
}

/**
 * Reads a shapefile.
 */
public void readShapeFile() {
	// select input file
    fileChooser.setFileFilter(new StandardFileFilter(new String[]{"shp"},"shape files"));
	int returnVal = fileChooser.showOpenDialog(this);
	if (returnVal != JFileChooser.APPROVE_OPTION)
		return;
	// determine shape file name
	String selectedName = fileChooser.getSelectedFile().getPath();
	File f = new File(selectedName);
	if (!f.exists()) {
		JOptionPane.showMessageDialog (this, "File "+selectedName+" does not exist!", "OK", JOptionPane.ERROR_MESSAGE); 
		return;
	}
	// reads the file in a thread
	readMenuItem.setEnabled(false);
	writeMenuItem.setEnabled(false);
	deleteMenuItem.setEnabled(false);
	new ShapeLoader(selectedName).start();
}

/**
 * Set resolution of network map.
 */
public void setResolution() {
	String resStr = JOptionPane.showInputDialog("Resolution of network file\nin x- and y-direction:", ""+resolution);
	if (resStr != null) {
		try {
			resolution = Integer.parseInt(resStr);
		} catch (Exception ex) {
		}
		if (resolution < 100)
			resolution = 100;
	}
}

/**
 * Writes the network files.
 */
public void writeNetworkFile() {
	readMenuItem.setEnabled(false);
	writeMenuItem.setEnabled(false);
	deleteMenuItem.setEnabled(false);
	new NetworkWriter().start();
}

}